import { supabase, supabaseAdmin, TABLES } from '@/config/supabase';
import { 
  Seller, 
  SellerInventory, 
  SellerOrder, 
  CreateSellerRequest, 
  UpdateSellerRequest,
  AddInventoryRequest,
  AssignOrderRequest,
  MarkFulfilledRequest,
  SellerWithInventory,
  SellerAnalytics
} from './types';

export const sellersService = {
  // Create new seller
  async createSeller(userId: string, sellerData: CreateSellerRequest): Promise<{ data?: Seller; error?: string }> {
    try {
      // Validate market exists if provided
      if (sellerData.market_id) {
        const { data: market, error: marketError } = await supabaseAdmin
          .from(TABLES.MARKETS)
          .select('id')
          .eq('id', sellerData.market_id)
          .single();

        if (marketError || !market) {
          return { error: 'Invalid market ID' };
        }
      }

      const { data, error } = await supabaseAdmin
        .from(TABLES.SELLERS)
        .insert({
          ...sellerData,
          created_by: userId,
          status: 'active'
        })
        .select()
        .single();

      if (error) throw error;

      // Send welcome notification (optional)
      try {
        // This would integrate with the notifications module
        console.log(`Welcome notification sent to seller: ${data.name}`);
      } catch (notifError) {
        console.log('Failed to send welcome notification:', notifError);
      }

      return { data };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // List sellers for user
  async listSellers(userId: string, filters?: { market_id?: string; status?: string }): Promise<{ data?: Seller[]; error?: string }> {
    try {
      let query = supabaseAdmin
        .from(TABLES.SELLERS)
        .select('*')
        .eq('created_by', userId);

      if (filters?.market_id) {
        query = query.eq('market_id', filters.market_id);
      }

      if (filters?.status) {
        query = query.eq('status', filters.status);
      }

      const { data, error } = await query.order('created_at', { ascending: false });

      if (error) throw error;
      return { data };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // Get seller by ID
  async getSeller(sellerId: string): Promise<{ data?: Seller; error?: string }> {
    try {
      const { data, error } = await supabaseAdmin
        .from(TABLES.SELLERS)
        .select('*')
        .eq('id', sellerId)
        .single();

      if (error) throw error;
      return { data };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // Update seller
  async updateSeller(sellerId: string, updateData: UpdateSellerRequest): Promise<{ data?: Seller; error?: string }> {
    try {
      const { data, error } = await supabaseAdmin
        .from(TABLES.SELLERS)
        .update(updateData)
        .eq('id', sellerId)
        .select()
        .single();

      if (error) throw error;
      return { data };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // Soft delete seller (mark as inactive)
  async deleteSeller(sellerId: string): Promise<{ data?: Seller; error?: string }> {
    try {
      const { data, error } = await supabaseAdmin
        .from(TABLES.SELLERS)
        .update({ status: 'inactive' })
        .eq('id', sellerId)
        .select()
        .single();

      if (error) throw error;
      return { data };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // Add or update inventory item
  async addInventory(sellerId: string, inventoryData: AddInventoryRequest): Promise<{ data?: SellerInventory; error?: string }> {
    try {
      // Check if item already exists
      const { data: existingItem } = await supabaseAdmin
        .from(TABLES.SELLER_INVENTORY)
        .select('*')
        .eq('seller_id', sellerId)
        .eq('product_name', inventoryData.product_name)
        .single();

      let result;
      if (existingItem) {
        // Update existing item
        const { data, error } = await supabaseAdmin
          .from(TABLES.SELLER_INVENTORY)
          .update({
            unit: inventoryData.unit,
            price: inventoryData.price,
            available: inventoryData.available
          })
          .eq('id', existingItem.id)
          .select()
          .single();

        if (error) throw error;
        result = data;
      } else {
        // Create new item
        const { data, error } = await supabaseAdmin
          .from(TABLES.SELLER_INVENTORY)
          .insert({
            seller_id: sellerId,
            ...inventoryData
          })
          .select()
          .single();

        if (error) throw error;
        result = data;
      }

      return { data: result };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // Get seller inventory
  async getInventory(sellerId: string): Promise<{ data?: SellerInventory[]; error?: string }> {
    try {
      const { data, error } = await supabaseAdmin
        .from(TABLES.SELLER_INVENTORY)
        .select('*')
        .eq('seller_id', sellerId)
        .order('created_at', { ascending: false });

      if (error) throw error;
      return { data };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // Update inventory item
  async updateInventory(itemId: string, updateData: UpdateInventoryRequest): Promise<{ data?: SellerInventory; error?: string }> {
    try {
      const { data, error } = await supabaseAdmin
        .from(TABLES.SELLER_INVENTORY)
        .update(updateData)
        .eq('id', itemId)
        .select()
        .single();

      if (error) throw error;
      return { data };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // Delete inventory item
  async deleteInventory(itemId: string): Promise<{ data?: any; error?: string }> {
    try {
      const { error } = await supabaseAdmin
        .from(TABLES.SELLER_INVENTORY)
        .delete()
        .eq('id', itemId);

      if (error) throw error;
      return { data: { success: true } };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // Assign seller to order
  async assignOrder(sellerId: string, orderData: AssignOrderRequest): Promise<{ data?: SellerOrder; error?: string }> {
    try {
      // Validate shopping list item exists
      const { data: listItem, error: itemError } = await supabaseAdmin
        .from(TABLES.SHOPPING_LIST_ITEMS)
        .select('id')
        .eq('id', orderData.shopping_list_item_id)
        .single();

      if (itemError || !listItem) {
        return { error: 'Invalid shopping list item ID' };
      }

      // Check if already assigned
      const { data: existingOrder } = await supabaseAdmin
        .from(TABLES.SELLER_ORDERS)
        .select('*')
        .eq('seller_id', sellerId)
        .eq('shopping_list_item_id', orderData.shopping_list_item_id)
        .single();

      if (existingOrder) {
        return { error: 'Seller already assigned to this item' };
      }

      const { data, error } = await supabaseAdmin
        .from(TABLES.SELLER_ORDERS)
        .insert({
          seller_id: sellerId,
          shopping_list_item_id: orderData.shopping_list_item_id,
          price: orderData.price,
          status: 'assigned',
          assigned_at: new Date().toISOString()
        })
        .select()
        .single();

      if (error) throw error;

      // Send notification to vendor (optional)
      try {
        console.log(`Order assigned notification sent for seller: ${sellerId}`);
      } catch (notifError) {
        console.log('Failed to send assignment notification:', notifError);
      }

      return { data };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // Mark order as fulfilled
  async markFulfilled(sellerId: string, fulfillmentData: MarkFulfilledRequest): Promise<{ data?: SellerOrder; error?: string }> {
    try {
      const { data, error } = await supabaseAdmin
        .from(TABLES.SELLER_ORDERS)
        .update({
          status: 'fulfilled',
          fulfilled_at: new Date().toISOString()
        })
        .eq('id', fulfillmentData.seller_order_id)
        .eq('seller_id', sellerId)
        .select()
        .single();

      if (error) throw error;

      // Send notification to vendor (optional)
      try {
        console.log(`Order fulfilled notification sent for seller: ${sellerId}`);
      } catch (notifError) {
        console.log('Failed to send fulfillment notification:', notifError);
      }

      return { data };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // Get seller analytics
  async getSellerAnalytics(sellerId: string): Promise<{ data?: SellerAnalytics; error?: string }> {
    try {
      // Get orders
      const { data: orders, error: ordersError } = await supabaseAdmin
        .from(TABLES.SELLER_ORDERS)
        .select('*')
        .eq('seller_id', sellerId);

      if (ordersError) throw ordersError;

      // Get inventory count
      const { data: inventory, error: inventoryError } = await supabaseAdmin
        .from(TABLES.SELLER_INVENTORY)
        .select('id')
        .eq('seller_id', sellerId);

      if (inventoryError) throw inventoryError;

      const totalOrders = orders?.length || 0;
      const fulfilledOrders = orders?.filter(o => o.status === 'fulfilled').length || 0;
      const totalRevenue = orders?.reduce((sum, order) => sum + (order.price || 0), 0) || 0;
      const averageOrderValue = totalOrders > 0 ? totalRevenue / totalOrders : 0;
      const inventoryCount = inventory?.length || 0;

      return { data: {
        total_orders: totalOrders,
        fulfilled_orders: fulfilledOrders,
        total_revenue: totalRevenue,
        average_order_value: averageOrderValue,
        inventory_count: inventoryCount
      }};
    } catch (error: any) {
      return { error: error.message };
    }
  }
}; 